<!DOCTYPE html>
<html lang="bn">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>নওরোজ । সত্য প্রকাশে নির্ভীক - ইউনিকোড-বিজয় কনভার্টার</title>
  
  <!-- Bootstrap 5 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  
  <!-- Font Awesome for icons -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  
  <!-- Bengali Fonts -->
  <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+Bengali:wght@400;500;700&display=swap" rel="stylesheet">
  
  <style>
    body {
      background-color: #f8f9fa;
      font-family: 'Noto Sans Bengali', 'Kalpurush', 'SolaimanLipi', Arial, sans-serif;
    }
    
    .header {
      padding: 20px 0;
      margin-bottom: 30px;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
      background-color: #fff;
    }
    
    .logo img {
      height: 80px;
      filter: drop-shadow(0 2px 4px rgba(0,0,0,0.3));
    }
    
    .section-title {
      font-size: 1.4rem;
      font-weight: bold;
      color: #0d47a1;
      margin-bottom: 15px;
      padding-bottom: 8px;
      border-bottom: 3px solid #0d47a1;
    }
    
    .converter-box {
      background-color: white;
      border-radius: 10px;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
      padding: 25px;
      margin-bottom: 30px;
      border: 1px solid #e0e0e0;
    }
    
    .text-area {
      width: 100%;
      min-height: 250px;
      padding: 15px;
      border: 2px solid #e0e0e0;
      border-radius: 8px;
      font-size: 1.1rem;
      font-family: 'Noto Sans Bengali', 'Kalpurush', Arial, sans-serif;
      resize: vertical;
      direction: ltr;
    }
    
    .text-area:focus {
      border-color: #0d47a1;
      box-shadow: 0 0 0 0.25rem rgba(13, 71, 161, 0.25);
      outline: none;
    }
    
    .btn-custom {
      font-size: 1rem;
      padding: 10px 25px;
      border-radius: 30px;
      margin: 0 5px 10px;
      font-weight: 600;
      transition: all 0.3s;
      border: none;
    }
    
    .btn-custom:hover {
      transform: translateY(-2px);
      box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
    }
    
    .btn-unicode-to-bijoy {
      background-color: #1976d2;
      color: white;
    }
    
    .btn-bijoy-to-unicode {
      background-color: #43a047;
      color: white;
    }
    
    .btn-clear {
      background-color: #e53935;
      color: white;
    }
    
    .conversion-icon {
      font-size: 1.5rem;
      color: #0d47a1;
      margin: 0 10px;
    }
    
    .action-buttons {
      display: flex;
      justify-content: center;
      flex-wrap: wrap;
      margin-top: 20px;
    }
    
    .chat-icon {
      position: fixed;
      bottom: 20px;
      right: 20px;
      width: 60px;
      height: 60px;
      background-color: #25d366;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      box-shadow: 0 4px 10px rgba(0, 0, 0, 0.3);
      z-index: 1000;
      transition: all 0.3s;
    }
    
    .chat-icon:hover {
      transform: scale(1.1);
      box-shadow: 0 6px 15px rgba(0, 0, 0, 0.4);
    }
    
    .chat-icon img {
      width: 35px;
    }
    
    .conversion-info {
      background-color: #e3f2fd;
      border-radius: 8px;
      padding: 15px;
      margin-top: 20px;
      font-size: 0.95rem;
    }
    
    @media (max-width: 768px) {
      .conversion-icon {
        display: none;
      }
      .text-area {
        min-height: 200px;
      }
    }
    
    .alert-custom {
      position: fixed;
      top: 100px;
      right: 20px;
      z-index: 1050;
      min-width: 300px;
    }
  </style>
</head>
<body>
  <!-- Header with Logo -->
  <div class="header">
    <div class="container">
      <div class="text-center">
        <div class="logo">
          <a href="#">
            <img src="https://monoputo.nawroj.com/frontend/assets/images/logonaw.png" alt="নওরোজ" class="img-fluid">
          </a>
        </div>
      </div>
    </div>
  </div>

  <div class="container">
    <!-- Converter Section -->
    <div class="converter-box">
      <div class="row">
        <!-- Unicode Section -->
        <div class="col-lg-6 mb-4">
          <h2 class="section-title">
            <i class="fas fa-keyboard me-2"></i>ইউনিকোড লেখা এখানে পেস্ট করুন
          </h2>
          <textarea id="unicodeText" class="text-area" placeholder="ইউনিকোড ফন্টে লিখিত বাংলা টেক্সট এখানে পেস্ট করুন..."></textarea>
          <div class="action-buttons">
            <button id="unicodeToBijoyBtn" class="btn btn-custom btn-unicode-to-bijoy">
              <i class="fas fa-arrow-right me-2"></i>ইউনিকোড থেকে বিজয়
            </button>
            <button id="clearUnicodeBtn" class="btn btn-custom btn-clear">
              <i class="fas fa-trash-alt me-2"></i>পরিষ্কার করুন
            </button>
          </div>
        </div>
        
        <!-- Arrow Icon for Visual Connection -->
        <div class="col-lg-12 d-lg-none text-center my-3">
          <i class="fas fa-exchange-alt conversion-icon fa-2x"></i>
        </div>
        <div class="col-lg-1 d-none d-lg-flex align-items-center justify-content-center">
          <i class="fas fa-exchange-alt conversion-icon fa-2x"></i>
        </div>
        
        <!-- Bijoy Section -->
        <div class="col-lg-5 mb-4">
          <h2 class="section-title">
            <i class="fas fa-keyboard me-2"></i>বিজয় লেখা এখানে পেস্ট করুন
          </h2>
          <textarea id="bijoyText" class="text-area" placeholder="বিজয় ফন্টে লিখিত বাংলা টেক্সট এখানে পেস্ট করুন..."></textarea>
          <div class="action-buttons">
            <button id="bijoyToUnicodeBtn" class="btn btn-custom btn-bijoy-to-unicode">
              <i class="fas fa-arrow-left me-2"></i>বিজয় থেকে ইউনিকোড
            </button>
            <button id="clearBijoyBtn" class="btn btn-custom btn-clear">
              <i class="fas fa-trash-alt me-2"></i>পরিষ্কার করুন
            </button>
          </div>
        </div>
      </div>
    </div>
  </div>

  <!-- Floating WhatsApp/Chat Icon -->
  <!--<a href="https://wa.me/8801700000000" target="_blank" class="chat-icon">-->
  <!--  <img src="https://upload.wikimedia.org/wikipedia/commons/6/6b/WhatsApp.svg" alt="Chat">-->
  <!--</a>-->

  <!-- Bootstrap JS Bundle with Popper -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  
  <script>
    // বিস্তৃত ইউনিকোড থেকে বিজয় ম্যাপিং টেবিল
    const unicodeToBijoyMap = {
      // স্বরবর্ণ
      'অ': 'Av', 'আ': 'A', 'ই': 'B', 'ঈ': 'C', 'উ': 'D', 'ঊ': 'E', 'ঋ': 'F', 'এ': 'G', 'ঐ': 'H', 'ও': 'I', 'ঔ': 'J',
      // ব্যঞ্জনবর্ণ
      'ক': 'K', 'খ': 'L', 'গ': 'M', 'ঘ': 'N', 'ঙ': 'O', 'চ': 'P', 'ছ': 'Q', 'জ': 'R', 'ঝ': 'S', 'ঞ': 'T',
      'ট': 'U', 'ঠ': 'V', 'ড': 'W', 'ঢ': 'X', 'ণ': 'Y', 'ত': 'Z', 'থ': '_', 'দ': '`', 'ধ': 'a', 'ন': 'b',
      'প': 'c', 'ফ': 'd', 'ব': 'e', 'ভ': 'f', 'ম': 'g', 'য': 'h', 'র': 'i', 'ল': 'j', 'শ': 'k', 'ষ': 'l',
      'স': 'm', 'হ': 'n', 'ড়': 'o', 'ঢ়': 'p', 'য়': 'q', 'ৎ': 'r', 'ং': 'M', 'ঃ': ':', 'ঁ': 'N',
      // কার (স্বরচিহ্ন)
      'া': 'v', 'ি': 'w', 'ী': 'x', 'ু': 'y', 'ূ': 'z', 'ৃ': '_', 'ে': 's', 'ৈ': 't', 'ো': 'u', 'ৌ': '~',
      // যুক্তাক্ষরের বিশেষ ম্যাপিং
      '্র': 'ª', 'র্': 'iæ', '্য': '«',
      // সংখ্যা
      '০': '0', '১': '1', '২': '2', '৩': '3', '৪': '4', '৫': '5', '৬': '6', '৭': '7', '৮': '8', '৯': '9',
      // যুক্তবর্ণের বিশেষ কেস
      'ক্': 'Kæ', 'খ্': 'Læ', 'গ্': 'Mæ', 'ঘ্': 'Næ', 'চ্': 'Pæ', 'ছ্': 'Qæ', 'জ্': 'Ræ', 'ঝ্': 'Sæ',
      'ট্': 'Uæ', 'ঠ্': 'Væ', 'ড্': 'Wæ', 'ঢ্': 'Xæ', 'ণ্': 'Yæ', 'ত্': 'Zæ', 'থ্': '_æ', 'দ্': '`æ',
      'ধ্': 'aæ', 'ন্': 'bæ', 'প্': 'cæ', 'ফ্': 'dæ', 'ব্': 'eæ', 'ভ্': 'fæ', 'ম্': 'gæ', 'য্': 'hæ',
      'র্': 'iæ', 'ল্': 'jæ', 'শ্': 'kæ', 'ষ্': 'læ', 'স্': 'mæ', 'হ্': 'næ',
      // বিশেষ যুক্তবর্ণ
      'ক্ষ': '¶', 'জ্ঞ': '¡', 'শ্র': 'kª', 'ঞ্ছ': 'Tz', 'ঞ্ঝ': 'TŠ', 'ট্ট': 'Uz', 'ড্ড': 'Wz', 'ণ্ঠ': 'Yz',
      'ন্থ': 'YæZ', 'ন্দ': 'b`', 'ন্ধ': 'ba', 'ম্প': 'gc', 'ম্ফ': 'gd', '্ল': 'jæ', '্র': 'ª'
    };

    // বিজয় থেকে ইউনিকোড ম্যাপিং টেবিল
    const bijoyToUnicodeMap = {
      // মৌলিক বর্ণ
      'Av': 'অ', 'A': 'আ', 'B': 'ই', 'C': 'ঈ', 'D': 'উ', 'E': 'ঊ', 'F': 'ঋ', 'G': 'এ', 'H': 'ঐ', 'I': 'ও', 'J': 'ঔ',
      'K': 'ক', 'L': 'খ', 'M': 'গ', 'N': 'ঘ', 'O': 'ঙ', 'P': 'চ', 'Q': 'ছ', 'R': 'জ', 'S': 'ঝ', 'T': 'ঞ',
      'U': 'ট', 'V': 'ঠ', 'W': 'ড', 'X': 'ঢ', 'Y': 'ণ', 'Z': 'ত', '_': 'থ', '`': 'দ', 'a': 'ধ', 'b': 'ন',
      'c': 'প', 'd': 'ফ', 'e': 'ব', 'f': 'ভ', 'g': 'ম', 'h': 'য', 'i': 'র', 'j': 'ল', 'k': 'শ', 'l': 'ষ',
      'm': 'স', 'n': 'হ', 'o': 'ড়', 'p': 'ঢ়', 'q': 'য়', 'r': 'ৎ', 'M': 'ং', ':': 'ঃ', 'N': 'ঁ',
      // কার
      'v': 'া', 'w': 'ি', 'x': 'ী', 'y': 'ু', 'z': 'ূ', '_': 'ৃ', 's': 'ে', 't': 'ৈ', 'u': 'ো', '~': 'ৌ',
      // যুক্তাক্ষর
      'ª': '্র', 'iæ': 'র্', '«': '্য',
      // সংখ্যা
      '0': '০', '1': '১', '2': '২', '3': '৩', '4': '৪', '5': '৫', '6': '৬', '7': '৭', '8': '৮', '9': '৯',
      // বিশেষ যুক্তবর্ণ
      '¶': 'ক্ষ', '¡': 'জ্ঞ', 'kª': 'শ্র'
    };

    // বিশেষ যুক্তবর্ণের জন্য প্যাটার্ন
    const specialConjuncts = [
      { pattern: /ক্ষ/g, replace: '¶' },
      { pattern: /জ্ঞ/g, replace: '¡' },
      { pattern: /শ্র/g, replace: 'kª' },
      { pattern: /ঞ্ছ/g, replace: 'Tz' },
      { pattern: /ঞ্ঝ/g, replace: 'TŠ' },
      { pattern: /ট্ট/g, replace: 'Uz' },
      { pattern: /ড্ড/g, replace: 'Wz' },
      { pattern: /ণ্ঠ/g, replace: 'Yz' },
      { pattern: /ন্থ/g, replace: 'YæZ' },
      { pattern: /ন্দ/g, replace: 'b`' },
      { pattern: /ন্ধ/g, replace: 'ba' },
      { pattern: /ম্প/g, replace: 'gc' },
      { pattern: /ম্ফ/g, replace: 'gd' }
    ];

    // ইউনিকোড থেকে বিজয়ে রূপান্তর ফাংশন
    function unicodeToBijoy(text) {
      let result = text;
      
      // প্রথমে বিশেষ যুক্তবর্ণগুলো রূপান্তর করুন
      for (const conjunct of specialConjuncts) {
        result = result.replace(conjunct.pattern, conjunct.replace);
      }
      
      // তারপর সাধারণ অক্ষরগুলো রূপান্তর করুন
      let converted = '';
      for (let i = 0; i < result.length; i++) {
        const char = result[i];
        const nextChar = result[i + 1];
        
        // ২-অক্ষরের বিশেষ কেস চেক করুন (যেমন: Av, iæ, etc.)
        if (nextChar && unicodeToBijoyMap[char + nextChar]) {
          converted += unicodeToBijoyMap[char + nextChar];
          i++; // পরের অক্ষরটাও নিয়ে নিয়েছি
        } else if (unicodeToBijoyMap[char]) {
          converted += unicodeToBijoyMap[char];
        } else {
          converted += char;
        }
      }
      
      return converted;
    }

    // বিজয় থেকে ইউনিকোডে রূপান্তর ফাংশন
    function bijoyToUnicode(text) {
      let result = text;
      
      // প্রথমে ২-অক্ষরের বিশেষ কেসগুলো চেক করুন
      let converted = '';
      for (let i = 0; i < result.length; i++) {
        const char = result[i];
        const nextChar = result[i + 1];
        const twoChars = char + nextChar;
        
        if (nextChar && bijoyToUnicodeMap[twoChars]) {
          converted += bijoyToUnicodeMap[twoChars];
          i++; // পরের অক্ষরটাও নিয়ে নিয়েছি
        } else if (bijoyToUnicodeMap[char]) {
          converted += bijoyToUnicodeMap[char];
        } else {
          converted += char;
        }
      }
      
      // বিশেষ যুক্তবর্ণগুলো রূপান্তর করুন
      converted = converted.replace(/¶/g, 'ক্ষ');
      converted = converted.replace(/¡/g, 'জ্ঞ');
      converted = converted.replace(/kª/g, 'শ্র');
      
      return converted;
    }

    // DOM Elements
    const unicodeTextarea = document.getElementById('unicodeText');
    const bijoyTextarea = document.getElementById('bijoyText');
    const unicodeToBijoyBtn = document.getElementById('unicodeToBijoyBtn');
    const bijoyToUnicodeBtn = document.getElementById('bijoyToUnicodeBtn');
    const clearUnicodeBtn = document.getElementById('clearUnicodeBtn');
    const clearBijoyBtn = document.getElementById('clearBijoyBtn');

    // Event Listeners
    unicodeToBijoyBtn.addEventListener('click', function() {
      const unicodeText = unicodeTextarea.value;
      const bijoyText = unicodeToBijoy(unicodeText);
      bijoyTextarea.value = bijoyText;
      showAlert('ইউনিকোড থেকে বিজয়ে রূপান্তর সম্পন্ন হয়েছে!', 'success');
    });

    bijoyToUnicodeBtn.addEventListener('click', function() {
      const bijoyText = bijoyTextarea.value;
      const unicodeText = bijoyToUnicode(bijoyText);
      unicodeTextarea.value = unicodeText;
      showAlert('বিজয় থেকে ইউনিকোডে রূপান্তর সম্পন্ন হয়েছে!', 'success');
    });

    clearUnicodeBtn.addEventListener('click', function() {
      unicodeTextarea.value = '';
      showAlert('ইউনিকোড টেক্সট পরিষ্কার করা হয়েছে!', 'info');
    });

    clearBijoyBtn.addEventListener('click', function() {
      bijoyTextarea.value = '';
      showAlert('বিজয় টেক্সট পরিষ্কার করা হয়েছে!', 'info');
    });

    // Function to show alert messages
    function showAlert(message, type) {
      // Remove any existing alert
      const existingAlert = document.querySelector('.alert-custom');
      if (existingAlert) {
        existingAlert.remove();
      }
      
      // Create alert element
      const alertEl = document.createElement('div');
      alertEl.className = `alert alert-${type === 'success' ? 'success' : 'info'} alert-dismissible fade show alert-custom`;
      alertEl.innerHTML = `
        <strong>${type === 'success' ? 'সফল!' : 'তথ্য:'}</strong> ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
      `;
      
      document.body.appendChild(alertEl);
      
      // Auto remove after 3 seconds
      setTimeout(() => {
        if (alertEl.parentNode) {
          alertEl.remove();
        }
      }, 3000);
    }

    // Sample text for demonstration - আপনার দেওয়া বাক্য
    window.addEventListener('load', function() {
      const sampleUnicodeText = "এই সফল অভিযানের ফলে কিজিলেলমা বিশ্বের একমাত্র অ-মানববিহীন প্ল্যাটফর্ম হিসেবে প্রতিষ্ঠিত হয়েছে - যার আকাশ থেকে আকাশে যুদ্ধের জন্য যাচাইকৃত ক্ষমতা রয়েছে।";
      const sampleBijoyText = "GB Dw›`n‡bi djvd‡b wK‡Rv‡jjgv wfwWI‡qi GKgvÎ A-‡gvbweevwnb cÖv‡`ªvb wn‡m‡e we¯—…wZ †c‡q‡Q - hv‡i Avk¨ †_‡K Avk¨‡Z hy×‡R †PŠ¤^vK…Z ¶gMv Av‡Q|";
      
      unicodeTextarea.value = sampleUnicodeText;
      bijoyTextarea.value = sampleBijoyText;
    });
  </script>
</body>
</html>